/*
 * cSimEngine.cpp
 *
 * THE simulated network layer underpinnings.
 */
#include "cSimEngine.h"
#include "cSimEndpoint.h"
#include "cSimNode.h"
#include "cSimNetLayerParam.h"
#include "cSimInfo.h"
#include "cSimMsgHeap.h"
#include "cSimMsgQueue.h"
#include "Interfaces/cLayer.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Protocol Stack/cMsgBuffer.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util/gError.h"
#include "Util/cException.h"

#include <stdlib.h>

#define MAX_DELIVER_IN_ONE_SCHEDULE 32

/*
 * cSimEngine::cSimEngine()
 * Purpose:	Sets up the simulation engine.
 * IN:		param	-> a cSimNetLayerParam that will define the simulation.
 *
 */
cSimEngine::cSimEngine(cSimNetLayerParam* param)
{
	mAddressCount = 0;
	mSubnetCount  = 0;
	mParam = *param;
	mNodes = new cSimNode*[mParam.mNumSubnets];
	if(!mNodes)
	{
		gError("Unable to set up subnet array.", __LINE__, __FILE__);
		throw cException("Unable to setup subnet array.");
	}
	for(unsigned int i = 0; i < mParam.mNumSubnets; i++)
	{
		mNodes[i] = new cSimNode[mParam.mNumAddressesPerSubnet];
		if(!mNodes[i])
		{
			gError("Unable to create address in subnet.", __LINE__, __FILE__);
			throw cException("Unable to create addresses.");
		}
	}
	mNetQueue = new cSimMsgHeap(mParam.mNumSubnets*mParam.mNumAddressesPerSubnet, 1024);
	mDeliverQueue = new cSimMsgQueue(mParam.mNumSubnets*mParam.mNumAddressesPerSubnet, 512);
	if(!mNetQueue || !mDeliverQueue)
	{
		gError("Unable to make net and deliver queues.", __LINE__, __FILE__);
		throw cException("Unable to create netsim queues.");
	}
}

cSimEngine::~cSimEngine()
{
	if(mNodes)
	{
		for(unsigned int i = 0; i < mParam.mNumSubnets; i++)
		{
			delete[] mNodes[i];
		}
		delete[] mNodes;
	}
	if(mNetQueue)	  { delete mNetQueue; }
	if(mDeliverQueue) { delete mDeliverQueue; }
}

/*
 * cSimEngine::GetFreshEndpoint()
 *
 * Purpose:	Allocate a fresh endpoint and node.
 * IN:		eventHandle	-> Handle to be set when data arrives for the given node.
 * OUT:		outEp		-> Pointer to be set that points to the node's new endpoint.
 * Cond:	There must be a node free for use.
 * PostCnd:	One less node is free for use.
 * Return:	true if success, else false.
 */
bool cSimEngine::GetFreshEndpoint(HANDLE eventHandle, cSimEndpoint** outEp)
{
	cSimEndpoint	*ep;
	if(mSubnetCount == mParam.mNumSubnets)
	{
		mSubnetCount = 0;
		mAddressCount++;
	}
	if(mAddressCount == mParam.mNumAddressesPerSubnet)
	{
		return false;	// All out of endpoints.
	}

	ep = (cSimEndpoint *)cEndpointFactory::AllocEndpoint(ENDPOINT_TYPE_SIM);
	if(!ep) { return false; }
	ep->AddRef();
	ep->mSubnetIndex  = mSubnetCount;
	ep->mAddressIndex = mAddressCount;
	mNodes[mSubnetCount][mAddressCount].mEndpoint	 = ep;
	mNodes[mSubnetCount][mAddressCount].mEventHandle = eventHandle;
	mSubnetCount++;
	*outEp = ep;
	return true;
}

/*
 * cSimEngine::Send()
 *
 * Purpose:	Creates a copy of the message, stamps it with a deliver time, and puts it in the priority queue.
 * IN:		ep			-> The destination endpoint. (ignored if multicast)
 *			buffer		-> The message to send.
 *			messageType	-> The type of message that is to be sent.
 * OUT:		
 * Cond:	There must be a node free for use.
 * PostCnd:	One less node is free for use.
 * Return:	true if success, else false.
 */
bool cSimEngine::Send(cSimEndpoint* sender, cSimEndpoint* dest, cMsgBuffer* buffer, unsigned int messageType)
{
	DWORD		 now = cProtocolStack::GetTime();
	bool		 sameSubnet;
	cMsgBuffer*  cloneBuffer;
	SimMsg		 msg;
	unsigned int numDest;
	unsigned int firstDest;
	unsigned int subnet = dest->mSubnetIndex;

	switch(messageType)
	{
		case MSG_TYPE_BCAST:
			firstDest = 0;
			numDest = mParam.mNumAddressesPerSubnet;
			sameSubnet = sender->IsSameSubnet(dest);
			break;

		case MSG_TYPE_MULTICAST:
			subnet = sender->mSubnetIndex;
			firstDest = 0;
			numDest = mParam.mNumAddressesPerSubnet;
			sameSubnet = true;			
			break;

		case MSG_TYPE_UNICAST:
			firstDest = dest->mAddressIndex;
			numDest  = firstDest+1;
			sameSubnet = sender->IsSameSubnet(dest);
			break;

		default:
			gError("Received unknown message type in send.", __LINE__, __FILE__);
	}

	// Prepare and send messages.
	for(unsigned int i = firstDest; i < numDest; i++)
	{
		// Get the destination.
		dest = mNodes[subnet][i].mEndpoint;
		if(!dest)
		{
			break;	// No more endpoints on this subnet
		}

		// Check if local loopback.
		if(messageType == MSG_TYPE_MULTICAST && !mParam.mRecvOwnMulticasts)
		{
			if(dest->Equals(sender))
			{
				continue;
			}
		}

		// Create a new buffer.
		cloneBuffer = buffer->Clone();
		if(!cloneBuffer)
		{
			gError("Unable to clone a buffer in the network simulation layer.", __LINE__, __FILE__);
			return false;
		}

		// Make sure these stick around.
		dest->AddRef();
		sender->AddRef();
		cloneBuffer->AddRef();

		// Set up the message.
		msg.type   = messageType;
		msg.destEp = dest;
		msg.sendEp = sender;
		msg.msg    = cloneBuffer;
		if(sameSubnet)
		{
			msg.timeToDeliver = now + mParam.mSubnetToSubnetLatency;
			msg.timeToDeliver += rand() % mParam.mSubnetToSubnetFudge;
		}
		else
		{
			msg.timeToDeliver = now + mParam.mInternalSubnetLatency;
			msg.timeToDeliver += rand() % mParam.mInternalSubnetFudge;
		}

		if(!mNetQueue->Insert(&msg))
		{
			gError("Unable to insert packet into network sim queue.", __LINE__, __FILE__);
		}
	}
	return true;
}	

/*
 * cSimEngine::GetMessage()
 *
 * Purpose:	Gets a message that is destined for delivery.
 * IN:		ep		-> The endpoint that thinks there is a message for it.
 *			outEp	-> The sender of the message.
 *			outBuf	-> The message buffer itself.
 *			type	-> The type of message that it was.
 *			-
 * OUT:		
 * Cond:	-
 * PostCnd:	Some events may be triggered.
 * Return:	true if success, else false.
 */
bool cSimEngine::GetMessage(cSimEndpoint* ep, cSimEndpoint** outEp, cMsgBuffer** outBuf, unsigned int* type)
{
	SimMsg	msg;
	if(mDeliverQueue->Find(ep, &msg))
	{
		*outEp  = msg.sendEp;
		*outBuf = msg.msg;
		*type	= msg.type;
		return true;
	}
	return false;
}

/*
 * cSimEngine::Schedule()
 *
 * Purpose:	Gives the engine a small timeslice to check for msgs that should be delivered.
 * IN:		-
 *			-
 *			-
 * OUT:		
 * Cond:	-
 * PostCnd:	Some events may be triggered.
 * Return:	true if success, else false.
 */
bool cSimEngine::Schedule()
{
	DWORD	now = cProtocolStack::GetTime();
	unsigned int numDeliver = 0;
	SimMsg	msg;

	while(mNetQueue->Peek(&msg) && 
		  msg.timeToDeliver < now   &&
		  (numDeliver < MAX_DELIVER_IN_ONE_SCHEDULE))
	{
		mNetQueue->Remove(&msg);		// Remove from the network queue.
		mDeliverQueue->Insert(&msg);	// Insert into deliver queue.
		SetEvent(mNodes[msg.destEp->mSubnetIndex][msg.destEp->mAddressIndex].mEventHandle);
		numDeliver++;
	}
	return true;
}
